/* batch_Homerun_H123_v2.jsx — Updates fill+stroke colors for baseball animations
 * Fixes:
 *  - Looser name matching for inner layers (HOME/RUN).
 *  - Handles both Text AND Shape layers inside precomps.
 *  - Default INNER_STROKE_NAMES includes TOUCH,DOWN,HOME,RUN.
 */
(function () {
  // --- load common helpers (force reload) ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) { try { delete $.global.GL; } catch(e) { $.global.GL = undefined; } $.evalFile(File(COMMON)); }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();
  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT     = GL.env("AE_PROJECT", null);
  var CSV_PATH    = GL.env("AE_CSV", null);
  var COMP_NAME   = GL.env("AE_COMP","Shatter-HR-192");

  // Layer names (sanitized)
  var LAYER_HOMERUN     = GL.cleanValue(GL.env("AE_LAYER_HOMERUN","HOMERUN"));
  var LAYER_HR_PRECOMP  = GL.cleanValue(GL.env("AE_LAYER_HR_PRECOMP","HomeRun PreComp"));
  var LAYER_LOGO        = GL.cleanValue(GL.env("AE_LAYER_LOGO","TeamLogo"));
  var LAYER_TEAMNAME    = GL.cleanValue(GL.env("AE_LAYER_TEAMNAME","TeamName"));

  var LAYER_HOMEPLATE = GL.cleanValue(GL.env("AE_LAYER_HOMEPLATE","HomePlateRunNum"));
  var HOMEPLATE_PATH  = GL.env("AE_HOMEPLATE_PATH","");


  // Inner names (now default includes HOME,RUN)
  var INNER_STROKE_NAMES = (GL.env("AE_INNER_STROKE_NAMES","TOUCH,DOWN,HOME,RUN")||"").split(",");

  // Stroke width
  var STROKE_W_STR = GL.env("AE_TEXT_STROKE_WIDTH","0.5");
  var STROKE_W     = (STROKE_W_STR!=="" && !isNaN(parseFloat(STROKE_W_STR))) ? parseFloat(STROKE_W_STR) : null;

  // Logos
  var logoOpts = { dir:GL.env("AE_LOGO_DIR",""),
                   tpl:GL.env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}"),
                   exts:GL.env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd") };

  // Output / templates
  var OUTDIR     = GL.env("AE_OUTDIR","");
  var PATH_TPL   = GL.env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME  = GL.env("AE_ANIM","H-1_2_3");
  var RS_TPL     = GL.env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL     = GL.env("AE_OM_TEMPLATE","PNG Sequence");

  // Controls
  var LEAGUE     = GL.env("AE_LEAGUE","");
  var LIMIT_STR  = GL.env("LIMIT","");
  var LIMIT      = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;
  var PURGE      = (GL.env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER  = (GL.env("AE_NO_RENDER","0")==="1");
  var QUIT_APP   = (GL.env("AE_QUIT","1")==="1");

  // --- open project & csv ---
  if(!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) GL.fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) GL.fail("AE_CSV env not set.");
  if(!LEAGUE || GL.cleanValue(LEAGUE)==="") GL.fail("AE_LEAGUE is required.");

  var rows   = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams  = GL.buildTeams(rows, LEAGUE);
  var todo   = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length>LIMIT) todo = todo.slice(0, LIMIT);
  if(!todo.length) GL.fail("No teams matched league: "+LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if(!comp) GL.fail("Comp not found: "+COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);
  GL.rqClear();

  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("Homerun H-1_2_3 - PNG Seq");

  // --- helpers ---
  function _safe01(c){ return GL.safeColor(GL.ensureNonBlack(c)); }
  function _normLoose(s){ return String(GL.cleanValue(s||"")).toLowerCase().replace(/[^a-z0-9]/g,""); }

    function _normLoose(s){ return String(GL.cleanValue(s||"")).toLowerCase().replace(/[^a-z0-9]/g,""); } // you already have this

    function replaceLayerFootageByNameDeep(compItem, targetName, filePath, depth){
    depth = depth||0; if (!compItem || !filePath) return 0;
    var want = _normLoose(targetName);
    var f = new File(filePath); if (!f.exists) { $.writeln("HomePlate file not found: "+filePath); return 0; }

    var io = new ImportOptions(f);
    if (!io.canImportAs(ImportAsType.FOOTAGE)) { $.writeln("Cannot import as footage: "+filePath); return 0; }
    io.importAs = ImportAsType.FOOTAGE;
    var footage = app.project.importFile(io);

    var hits = 0;
    for (var i=1;i<=compItem.numLayers;i++){
        var L = compItem.layer(i);
        try{
        var nm = _normLoose(L.name);
        if (nm === want && L.source){ // only layers with a source can be replaced
            try { L.replaceSource(footage, false); hits++; } catch(e){}
        }
        // dive into precomps
        if (L.source && (L.source instanceof CompItem)){
            hits += replaceLayerFootageByNameDeep(L.source, targetName, filePath, depth+1);
        }
        }catch(e){}
    }
    return hits;
    }


  // remove text-animator props that override color/width
  function _killTextColorAnimators(textLayer){
    try{
      var TP = textLayer.property("ADBE Text Properties");
      var ANIMS = TP && TP.property("ADBE Text Animators");
      if (!ANIMS) return 0;
      var removed=0;
      for (var ai=ANIMS.numProperties; ai>=1; ai--){
        var A = ANIMS.property(ai);
        var PROPS = A && A.property("ADBE Text Animator Properties");
        if (!PROPS) continue;
        var hasOverride = PROPS.property("ADBE Text Fill Color") || PROPS.property("ADBE Text Stroke Color") || PROPS.property("ADBE Text Stroke Width");
        if (hasOverride){ A.remove(); removed++; }
      }
      return removed;
    }catch(e){ return 0; }
  }

  // sync Effects (Fill/Tint)
  function _syncEffects(layer, color01){
    var fx = layer.property("Effects"); if (!fx) return;
    for (var i=1;i<=fx.numProperties;i++){
      var eff = fx.property(i); if (!eff) continue;
      if (eff.matchName === "ADBE Fill"){
        for (var j=1;j<=eff.numProperties;j++){
          var p = eff.property(j); if (!p) continue;
          try{
            if (p.propertyValueType === PropertyValueType.COLOR) p.setValue(color01);
            if (p.matchName === "ADBE Fill-0006") p.setValue(100);      // Opacity
            if (p.matchName === "ADBE Fill-0007") p.setValue(0);        // Blend w/ Original
          }catch(e){}
        }
      }
      if (eff.matchName === "ADBE Tint"){
        for (var k=1;k<=eff.numProperties;k++){
          var t = eff.property(k); if (!t) continue;
          try{
            if (t.propertyValueType === PropertyValueType.COLOR) t.setValue(color01);
            if (t.propertyValueType === PropertyValueType.ONE_D && /Amount/i.test(t.name)) t.setValue(100);
          }catch(e){}
        }
      }
    }
  }

  // sync Layer Styles (Color Overlay/Stroke)
  function _syncLayerStyles(layer, color01){
    var ls = layer.property("ADBE Layer Styles") || layer.property("Layer Styles");
    if (!ls) return;
    var co = ls.property("ADBE Color Overlay") || ls.property("Color Overlay");
    if (co){
      var coColor   = co.property("ADBE Color Overlay-0002") || co.property("Color");
      var coOpacity = co.property("ADBE Color Overlay-0003") || co.property("Opacity");
      var coEnabled = co.property("ADBE Color Overlay-0001") || co.property("enabled");
      try{ if (coEnabled) coEnabled.setValue(true); if (coColor) coColor.setValue(color01); if (coOpacity) coOpacity.setValue(100); }catch(e){}
    }
    var st = ls.property("ADBE Stroke") || ls.property("Stroke");
    if (st){
      var sColor = st.property("ADBE Stroke-0003") || st.property("Color");
      try{ if (sColor) sColor.setValue(color01); }catch(e){}
    }
  }

  // TEXT: fill + stroke
  function setTextFillAndStroke(layer, color01, strokeW){
    if (!layer) return false;
    var st = layer.property("Source Text"); if (!st) return false;
    var c = _safe01(color01);
    try{
      var removed = _killTextColorAnimators(layer);
      var td = st.value;
      td.applyFill   = true; td.fillColor   = c;
      td.applyStroke = true; td.strokeColor = c;
      if (strokeW !== null && strokeW !== undefined) td.strokeWidth = strokeW;
      try{ td.strokeOverFill = false; }catch(e){}
      st.setValue(td);
      _syncEffects(layer, c);
      _syncLayerStyles(layer, c);
      return true;
    }catch(e){ $.writeln("setTextFillAndStroke ERR: "+e); return false; }
  }

  // SHAPES: fill + stroke (recursively)
  function setShapeFillAndStroke(layer, color01, strokeW){
    if (!layer) return 0;
    var c = _safe01(color01);
    var contents = layer.property("ADBE Root Vectors Group");
    if (!contents) return 0;
    function walk(group){
      var hits=0;
      for (var i=1;i<=group.numProperties;i++){
        var p = group.property(i);
        if (!p) continue;
        var mn = p.matchName;
        if (mn === "ADBE Vector Group" || mn === "ADBE Vector Filter" || mn === "ADBE Vector Shape - Group"){
          hits += walk(p);
        } else if (mn === "ADBE Vector Graphic - Fill"){
          var fc = p.property("ADBE Vector Fill Color");
          var fo = p.property("ADBE Vector Fill Opacity");
          try{ if (fc) fc.setValue(c); if (fo) fo.setValue(100); hits++; }catch(e){}
        } else if (mn === "ADBE Vector Graphic - Stroke"){
          var sc = p.property("ADBE Vector Stroke Color");
          var so = p.property("ADBE Vector Stroke Opacity");
          var sw = p.property("ADBE Vector Stroke Width");
          try{
            if (sc) sc.setValue(c);
            if (so) so.setValue(100);
            if (sw && strokeW!==null && strokeW!==undefined) sw.setValue(strokeW);
            hits++;
          }catch(e){}
        }
      }
      return hits;
    }
    return walk(contents);
  }

  // Deep updater by NAME: updates Text AND Shape layers; recurses into precomps
  function setFillAndStrokeOnNamedLayersDeep(compItem, namesArr, color01, strokeW, depth){
    depth = depth || 0; if (depth>12 || !compItem) return 0;
    var targets = {};
    for (var n=0;n<namesArr.length;n++){ targets[_normLoose(namesArr[n])] = true; }

    $.writeln("\n[setFillAndStrokeOnNamedLayersDeep] Comp: " + compItem.name + " @depth " + depth);
    var hits=0;
    for (var i=1;i<=compItem.numLayers;i++){
      var L = compItem.layer(i);
      try{
        var lname = _normLoose(L.name);
        if (targets[lname]){
          $.writeln("  -> MATCH: '"+L.name+"' ("+L.matchName+")");
          if (L.matchName === "ADBE TextLayer"){
            if (setTextFillAndStroke(L, color01, strokeW)) hits++;
          } else if (L.matchName === "ADBE Vector Layer"){
            hits += setShapeFillAndStroke(L, color01, strokeW);
          } else {
            // still try effects/layer styles if present
            _syncEffects(L, _safe01(color01));
            _syncLayerStyles(L, _safe01(color01));
          }
        }
        // Recurse into precomps
        if (L.source && (L.source instanceof CompItem)){
          hits += setFillAndStrokeOnNamedLayersDeep(L.source, namesArr, color01, strokeW, depth+1);
        }
      }catch(e){ $.writeln("  Layer err: "+e); }
    }
    $.writeln("  Hits @depth "+depth+": "+hits);
    return hits;
  }

  function setTeamNameTextAndStroke(compItem, nameStr, strokeColor01, strokeW){
    $.writeln("\n[setTeamNameTextAndStroke] -> "+nameStr);
    var lyr = GL.getLayer(compItem, LAYER_TEAMNAME);
    if (!lyr) lyr = (function(c,t){
      var want=_normLoose(t);
      for (var i=1;i<=c.numLayers;i++){ try{
        if (_normLoose(c.layer(i).name)===want) return c.layer(i);
      }catch(e){} }
      return null;
    })(compItem, LAYER_TEAMNAME);
    if (!lyr){ $.writeln("  WARN: TeamName layer not found"); return 0; }

    GL.setTextContent(lyr, nameStr||"");

    var st = lyr.property("Source Text"); if (!st) return 0;
    try{
      var td = st.value;
      td.applyStroke = true;
      td.strokeColor = _safe01(strokeColor01);
      if (strokeW!==null && strokeW!==undefined) td.strokeWidth = strokeW;
      st.setValue(td);
      return 1;
    }catch(e){ $.writeln("  ERR set team name: "+e); return 0; }
  }

  // --- per team ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("========================================");

  for (var i=0;i<todo.length;i++){
    var t = todo[i];


    var smart = GL.computeSmartColors( GL.safeColor(t.primary), GL.safeColor(t.secondary) );
    var P = smart.primary;
    var S = smart.secondary;

    // 1) HOMERUN — Text fill+stroke = PRIMARY
    $.writeln("\n[1] HOMERUN layer...");
    var hr = GL.getLayer(comp, LAYER_HOMERUN);
    // if (hr && hr.matchName==="ADBE TextLayer"){
      setTextFillAndStroke(hr, P, STROKE_W);
    // } else {
    //   $.writeln("WARN: HOMERUN not found as TextLayer");
    // }

    // 2) INNER (TOUCH/DOWN/HOME/RUN) — Text/Shape fill+stroke = PRIMARY
    $.writeln("\n[2] Precomp inner layers (TOUCH/DOWN/HOME/RUN)...");
    var hrPre = GL.getLayer(comp, LAYER_HR_PRECOMP);
    if (hrPre && hrPre.source && (hrPre.source instanceof CompItem)){
      var innerHits = setFillAndStrokeOnNamedLayersDeep(hrPre.source, INNER_STROKE_NAMES, P, STROKE_W, 0);
      $.writeln("  Inner hits total: " + innerHits);
    } else {
      $.writeln("WARN: Precomp not found: " + LAYER_HR_PRECOMP);
    }

    // 2b) Swap HomePlateRunNum image if provided
    if (HOMEPLATE_PATH && LAYER_HOMEPLATE){
    var swapped = replaceLayerFootageByNameDeep(comp, LAYER_HOMEPLATE, HOMEPLATE_PATH, 0);
    $.writeln("HomePlate replacements: " + swapped + "  (" + HOMEPLATE_PATH + ")");
    }


    // 3) Logo replace
    $.writeln("\n[3] Team logo...");
    if (GL.replaceLogo(comp, LAYER_LOGO, t.league, t.abbr, logoOpts)) $.writeln("  Logo replaced."); else $.writeln("  Logo replace failed.");

    // 4) Team name + secondary stroke
    $.writeln("\n[4] Team name + stroke...");
    setTeamNameTextAndStroke(comp, t.name, S, STROKE_W);

    // 5) Pixel snap
    $.writeln("\n[5] Pixel snap...");
    var snapped = GL.snapCompTextForPixelArt(comp);
    $.writeln("  Snapped: " + snapped);

    if (PURGE && app.purge){ $.writeln("\n[6] Purge caches..."); try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

    // 7) Render
    if (!NO_RENDER){
      $.writeln("\n[7] Render...");
      var lc    = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    $.writeln("\n==== TEAM " + t.abbr + " COMPLETE ====");
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }

  $.writeln("\n\n========================================");
  $.writeln("ALL TEAMS PROCESSED");
  $.writeln("========================================");

  if (QUIT_APP) app.quit();
})();
